﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;

public class Shield : MonoBehaviour
{
    public CanvasGroup m_CanvasGroup;
    public AudioSource m_AudioSourceLevelUpDown;
    public AudioSource m_AudioSourceVibe;

    public Image[] m_SelectedArrows;

    public MeshCollider m_MeshCollider;
    public MeshFilter m_MeshFilter;
    Mesh m_Mesh;


    private Quaternion m_TargetRotation;
    public bool m_IsOnline = false;
    private Quaternion m_StartRotation;

    public float m_Angle;

    void Awake()
    {
        m_Mesh = m_MeshFilter.sharedMesh;
        m_TargetRotation = transform.localRotation;
        m_StartRotation = transform.localRotation;
    }
    public void SetAngle(float angle)
    {
        m_Angle = angle;
        m_TargetRotation = Quaternion.Euler(0, 0, m_Angle);
    }

    private void Update()
    {
        float angleDif = Mathf.Abs(m_TargetRotation.eulerAngles.z - transform.localRotation.eulerAngles.z);
        if (angleDif>0.1f)
        {
            Quaternion rot = Quaternion.Lerp(transform.localRotation, m_TargetRotation, (Mathf.Min(1,Time.deltaTime*10)));
            transform.localRotation = rot;
            m_MeshCollider.sharedMesh = m_Mesh;
            m_MeshCollider.enabled = false;
            m_MeshCollider.enabled = true;
        }

    }


    public void OnTriggerEnter(Collider other)
    {
        if (Game.Instance.m_Panic)
            return;

        Asteroid asteroid = other.gameObject.GetComponentInParent<Asteroid>();
        if (asteroid != null)
        {
            if (!asteroid.m_CollidedShields.Contains(this))
            {
                if (!asteroid.m_HasBeenAbsorbed)
                {
                    Game.Instance.AddEnergy(1);
                    
                }
                SpawnHitEffect(asteroid);
                asteroid.ReduceLevel(1);
                asteroid.m_CollidedShields.Add(this);
            }
            
        }
    }

    private void SpawnHitEffect(Asteroid asteroid)
    {
        
        AsteroidHitEffect hitEffect = Instantiate(Game.Instance.m_AsteroidHitEffectPrefab) as AsteroidHitEffect;
        hitEffect.transform.SetParent(Game.Instance.m_AsteroidSpawner.m_HitEffectsRoot);
        hitEffect.transform.position = asteroid.transform.position;
        AudioClip hitAudioClip = Sounds.Instance.GetShieldHitAudioClip();
        hitEffect.Initialize(hitAudioClip, asteroid);

    }

    public void PutOnline()
    {
        gameObject.SetActive(true);
        m_IsOnline = true;
        m_CanvasGroup.gameObject.SetActive(true);
        m_MeshCollider.enabled = true;
        m_AudioSourceLevelUpDown.clip = Sounds.Instance.GetShieldLevelUpAudioClip();
        m_AudioSourceLevelUpDown.Play();

        if(Game.Instance.m_SelectedShield == this)
        {
            Select();
        }
        else
        {
            Deselect();
        }

        StopAllCoroutines();
        StartCoroutine(FadeInCoroutine());

    }
    public void PutOffline(bool playSound=true)
    {
        m_IsOnline = false;

        m_CanvasGroup.gameObject.SetActive(true);
        m_MeshCollider.enabled = false;
        if (playSound)
        {
            m_AudioSourceLevelUpDown.clip = Sounds.Instance.GetShieldLevelDownAudioClip();
            m_AudioSourceLevelUpDown.Play();
        }
        StopAllCoroutines();
        if (gameObject.activeInHierarchy)
        {
            StartCoroutine(FadeOutCoroutine());
        }
    }


    private IEnumerator FadeInCoroutine()
    {
        m_CanvasGroup.gameObject.SetActive(true);
        m_MeshCollider.enabled = true;
        float a = 0;
        m_CanvasGroup.alpha = a;
        float duration = 0.5f;
        float timer = 0;
        float p = 0;
        while (p < 1)
        {
            timer += Time.deltaTime;
            p = timer / duration;
            a = p;
            m_CanvasGroup.alpha = a;
            yield return null;
        }
        m_CanvasGroup.alpha = 1;
    }

    private IEnumerator FadeOutCoroutine()
    {
        m_CanvasGroup.gameObject.SetActive(true);
        m_MeshCollider.enabled = true;
        float a = m_CanvasGroup.alpha;
        float startAlpha = a;
        float targetAlpha = 0;
        float duration = 0.3f;
        float timer = 0;
        float p = 0;
        while (p < 1)
        {
            timer += Time.deltaTime;
            p = timer / duration;
            a = Mathf.Lerp(startAlpha, targetAlpha, p);
            m_CanvasGroup.alpha = a;
            yield return null;
        }
        m_CanvasGroup.alpha = 0;
        m_CanvasGroup.gameObject.SetActive(false);
        m_MeshCollider.enabled = false;
    }

    public void Reset()
    {
        transform.localRotation = m_StartRotation;
        m_TargetRotation = m_StartRotation;
    }

    public void Select()
    {
        for (int i = 0; i < m_SelectedArrows.Length; i++)
        {
            m_SelectedArrows[i].gameObject.SetActive(true);
        }
    }
    public void Deselect()
    {
        for (int i = 0; i < m_SelectedArrows.Length; i++)
        {
            m_SelectedArrows[i].gameObject.SetActive(false);
        }
    }

}
