﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;
using TMPro;
using System.Globalization;

public class Game : MonoBehaviour
{
    public static Game Instance;
    public bool m_DebugControls = false;

    [Header("Game Components")]
    public Spaceship m_SpaceShip;
    public AsteroidSpawner m_AsteroidSpawner;
    public LastShield m_LastShield;
    public AsteroidHitEffect m_AsteroidHitEffectPrefab;
    public GameObject m_SpaceShipRoot;
    public AudioSource m_Music;

    [Header("UI")]
    public Slider m_EnergyBar;
    public TMP_Text m_LevelText;
    public TMP_Text m_EnergyText;
    public Button[] m_RestartButtons;
    public Button[] m_StartButtons;
    public Animator m_StartScreenAnimator;
    public Animator m_ExplosionAnimator;
    public Animator m_ScoreCanvasAnimator;
    public GameObject m_CanvasEndScreen;
    public CanvasGroup m_EndScreenCenterZone;
    public TMP_Text m_ScoreEndScreenText;
    public TMP_Text m_ScoreText;
    public GameObject m_TutoSouris;
    public GameObject m_TutoJoystick;
    public StartLevelSelector m_StartLevelSelector;
    public ReStartLevelSelector m_RestartLevelSelector;

    public Image m_JaugeImage;
    public Sprite[] m_JaugeLevels;



    [Header("Game Parameters")]
    public int m_StartEnergy = 3;
    public int m_StartLevel = 1;

    public DifficultyLevel[] m_DifficultyLevels;

    public float[] m_SpeedLimitForArrows;

    [Header("Debug")]
    public bool m_JostickPresent;

    public int m_DifficultyLevelId = 0;
    public Level[] m_Levels;
    public AnimationCurve m_DifficultyMultiplicatorOverTimeCurve;
    public float m_Energy = 3;
    public int m_MaxEnergy = 10;
    public int m_LevelId = 1; // Number of Shields
    public float m_Score = 0;
    public float m_ScoreBonus = 0;
    public Camera m_Camera;

    public Shield m_SelectedShield;
    private float m_SelectedShieldAngle = 0;
    private float m_StartGameTime;
    public float ElapsedGameTime { get { return Time.realtimeSinceStartup - m_StartGameTime; } }
    public float DifficultyMultiplicatorOverTime { get { return Game.Instance.m_DifficultyMultiplicatorOverTimeCurve.Evaluate(Game.Instance.ElapsedGameTime); } }
    public float m_BurstMultiplier = 1;


    private AudioSource m_AudioSource;
    private bool MaxLevel { get { return m_LevelId >= m_Levels.Length; } }
    public bool m_GameOver = false;
    public bool m_GameStarted = false;

    public bool GameRunning { get { return !m_GameOver && m_GameStarted; } }

    public bool m_IsSpeedFast = false;

    private Scrap[] m_Scraps;
    NumberFormatInfo nfi = (NumberFormatInfo)CultureInfo.InvariantCulture.NumberFormat.Clone();

    public bool m_Panic = false;

    public Level CurrentLevel
    {
        get
        {
            if (m_LevelId > 0)
            {
                return m_Levels[m_LevelId - 1];
            }
            else
            {
                return m_Levels[0];
            }
        }
    }

    public int m_MaxLevelIdReached;

    public Level m_MaxLevelReached
    {
        get
        {
            if (m_MaxLevelIdReached > 0)
            {
                return m_Levels[m_MaxLevelIdReached - 1];
            }
            else
            {
                return m_Levels[0];
            }
        }
    }





       
    void Awake()
    {
        Instance = this;
        m_Camera = Camera.main;
        m_AudioSource = GetComponent<AudioSource>();

        m_StartLevelSelector.Initialize();
        m_RestartLevelSelector.Initialize();
    }
    void Start()
    {
        // Joysticks
        string[] joysticks = Input.GetJoystickNames();
        for (int i = 0; i < joysticks.Length; i++)
        {
            if (joysticks[i].Length > 0)
            {
                m_JostickPresent = true;
            }
        }
        m_TutoSouris.gameObject.SetActive(!m_JostickPresent);
        m_TutoJoystick.gameObject.SetActive(m_JostickPresent);

        
        nfi.NumberGroupSeparator = " ";

        m_RestartButtons[0].onClick.AddListener(StartGame_Easy);
        m_RestartButtons[1].onClick.AddListener(StartGame_Medium);
        m_RestartButtons[2].onClick.AddListener(StartGame_Hard);

        m_StartButtons[0].onClick.AddListener(StartGame_Easy);
        m_StartButtons[1].onClick.AddListener(StartGame_Medium);
        m_StartButtons[2].onClick.AddListener(StartGame_Hard);
        SetUIButtonsInteractable(false);
        m_StartLevelSelector.gameObject.SetActive(false);
        StartCoroutine(SelectButtonCoroutine());

        m_Scraps = m_EndScreenCenterZone.GetComponentsInChildren<Scrap>(true);
        InitializeScraps();
        m_SelectedShield = m_SpaceShip.m_Shields[0];
        m_SelectedShield.Select();
        m_SpaceShipRoot.SetActive(true);
        m_StartScreenAnimator.gameObject.SetActive(true);
        m_CanvasEndScreen.gameObject.SetActive(false);
        
        RefreshUI();
    }


    private IEnumerator SelectButtonCoroutine()
    {
        
        yield return new WaitForSeconds(1);
        SetUIButtonsInteractable(true);
        m_StartLevelSelector.gameObject.SetActive(true);
        m_StartButtons[1].Select();
    }



    void Update()
    {
        if (Application.isEditor && m_DebugControls)
        {
            if (Input.GetKeyDown(KeyCode.S))
            {
                StartGame();
            }
        }

        if (!GameRunning)
        {
            return;
        }
        ManageMouseInputs();
        ManageJoystickInput();
        if (Application.isEditor && m_DebugControls)
        {
            DebugControls();
        }

        //Debug.Log("DifficultyMultiplicatorOverTime : " + DifficultyMultiplicatorOverTime);

        if (m_EnergyBar.value != m_Energy)
        {
            m_EnergyBar.value = Mathf.Lerp(m_EnergyBar.value, m_Energy, Time.deltaTime * 5);
        }

    }

    void DebugControls()
    {
        if (Input.GetKeyDown(KeyCode.A))
        {
            m_AsteroidSpawner.Spawn();
        }

        if (Input.GetKeyDown(KeyCode.KeypadPlus))
        {
            AddEnergy(1);
        }
        if (Input.GetKeyDown(KeyCode.KeypadMinus))
        {
            AddEnergy(-1);
        }
        if (Input.GetKeyDown(KeyCode.Keypad2))
        {
            if (Input.GetKey(KeyCode.LeftControl))
            {
                AddEnergy(-2);
            }
            else
            {
                AddEnergy(2);
            }
        }
        if (Input.GetKeyDown(KeyCode.Keypad3))
        {
            if (Input.GetKey(KeyCode.LeftControl))
            {
                AddEnergy(-3);
            }
            else
            {
                AddEnergy(3);
            }
        }

    }


    float GetAngleFromMousePos()
    {
        Vector3 clickDirection = m_Camera.ScreenToWorldPoint(Input.mousePosition).normalized;
        //Debug.Log("ClickDirection : " + clickDirection + "   Input.mousePosition"  + Input.mousePosition);
        clickDirection.z = 0;
        float angle = -(Vector3.Angle(Vector3.up, clickDirection) - 180);
        float dot = Vector3.Dot(Vector3.left, clickDirection);
        float dotSign = Mathf.Sign(dot);
        angle = angle * dotSign;
        return angle;
    }

    private float GetAngleFromJoysticks(float h, float v)
    {
        Vector3 inputDirection = new Vector3(h, v, 0);
        float angle = Vector3.Angle(inputDirection.normalized, Vector3.up);

        if (h > 0)
        {
            angle= 360 - angle;
        }
        return angle;
    }


    void ManageMouseInputs()
    {

        if (Input.GetMouseButtonDown(0))
        {
            Ray ray = m_Camera.ScreenPointToRay(Input.mousePosition);
            RaycastHit hit;
            if (Physics.Raycast(ray, out hit))
            {
                Shield shield = hit.collider.GetComponent<Shield>();
                if (shield != null && shield!=m_SelectedShield)
                {
                    if (m_SelectedShield != null)
                    {
                        m_SelectedShield.Deselect();
                    }
                    m_SelectedShield = shield;
                    m_SelectedShield.Select();
                    m_SelectedShieldAngle = GetAngleFromMousePos();
                }
            }
        }
        if (Input.GetMouseButtonUp(0))
        {
            //m_SelectedShield = null;
        }
        if(Input.GetMouseButton(0) && m_SelectedShield!=null)
        {
            float angle = GetAngleFromMousePos();
            m_SelectedShield.SetAngle(angle);
        }
        if (Input.GetMouseButton(1))
        {
            float angle = GetAngleFromMousePos();
            SetAllShieldsToAngle(angle);
        }
    }

    private void ManageJoystickInput()
    {
        if (m_JostickPresent)
        {
            if (Input.GetButtonDown("Joystick 1"))
            {
                ChangeShield();
            }

            if (Input.GetButton("Joystick 2"))
            {
                if (m_SelectedShield != null)
                {
                    SetAllShieldsToAngle(m_SelectedShield.m_Angle);
                }
            }

            float horizontalAxis = Input.GetAxisRaw("StickHorizontal");
            float verticalAxis = Input.GetAxisRaw("StickVertical");

            if (horizontalAxis != 0 || verticalAxis != 0)
            {
                float angle = GetAngleFromJoysticks(horizontalAxis, verticalAxis);
                m_SelectedShield.SetAngle(angle);


            }

        }
    }

    private void ChangeShield()
    {
        Shield[] activeShieldIds = GetActiveShields();
        if (activeShieldIds.Length > 1)
        {
            int activeShieldId = GetShieldId(m_SelectedShield, activeShieldIds);
            if (activeShieldId != -1)
            {
                activeShieldIds[activeShieldId].Deselect();
                activeShieldId = (int)Mathf.Repeat(activeShieldId + 1, activeShieldIds.Length);
                activeShieldIds[activeShieldId].Select();
                m_SelectedShield = activeShieldIds[activeShieldId];
            }
        }
        
    }
    
    private int GetShieldId(Shield shield, Shield[] activeShields)
    {
        for (int i = 0; i < activeShields.Length; i++)
        {
            if(activeShields[i] == shield)
            {
                return i;
            }
        }
        return -1;
    }
    private Shield[] GetActiveShields()
    {
        List<Shield> activeShieldIds = new List<Shield>();
        for (int i = 0; i < m_SpaceShip.m_Shields.Length; i++)
        {
            if (m_SpaceShip.m_Shields[i].m_IsOnline)
            {
                activeShieldIds.Add(m_SpaceShip.m_Shields[i]);
            }
        }
        return activeShieldIds.ToArray();
    }

    private void SetAllShieldsToAngle(float angle)
    {
        for (int i = 0; i < m_SpaceShip.m_Shields.Length; i++)
        {
            Shield shield = m_SpaceShip.m_Shields[i];
            if (shield.m_IsOnline)
            {
                m_SpaceShip.m_Shields[i].SetAngle(angle);
            }
        }
    }


    public void StartGame_Easy()
    {
        Debug.Log("StartGame_Easy");
        StartGame(0);
    }

    public void StartGame_Medium()
    {
        Debug.Log("StartGame_Medium");
        StartGame(1);
    }

    public void StartGame_Hard()
    {
        Debug.Log("StartGame_Hard");
        StartGame(2);
    }

    public void StartGame(int difficultyLevelId=0)
    {
        SetUIButtonsInteractable(false);
        m_DifficultyLevelId = difficultyLevelId;
        m_Levels = m_DifficultyLevels[difficultyLevelId].m_Levels;
        m_DifficultyMultiplicatorOverTimeCurve = m_DifficultyLevels[difficultyLevelId].m_DifficultyMultiplicatorOverTimeCurve;
        //Debug.Log("diff: " + difficultyLevelId  + " AZ : "+ m_DifficultyMultiplicatorOverTimeCurve.Evaluate(0));
        m_BurstMultiplier = m_DifficultyLevels[difficultyLevelId].m_BurstMultiplier;

        StopAllCoroutines();
        m_IsSpeedFast = true;
        m_Score = 0;
        m_SpaceShipRoot.SetActive(true);
        m_StartGameTime = Time.realtimeSinceStartup;
        m_EnergyBar.value = 0;
        m_Panic = false;

        if (m_GameStarted)
        {
            m_Music.Play();
            m_SpaceShipRoot.transform.position = new Vector3(-11, 0, 0);
            StartCoroutine(RestartGameCoroutine());
        }
        else
        {
            m_StartScreenAnimator.SetTrigger("FadeOut");
            m_GameOver = false;
            m_SpaceShipRoot.transform.position = Vector3.zero;
            m_LevelId = m_StartLevel;
        }


        m_GameStarted = true;
        
        m_Energy = m_StartEnergy;
        m_MaxEnergy = CurrentLevel.m_EnergyToComplete;
        m_MaxLevelIdReached = 0;
        m_AsteroidSpawner.StartSpawn();
        m_SpaceShip.ResetShields();
        m_SpaceShip.RefreshShields();
        m_LastShield.Reset();

        m_ScoreCanvasAnimator.gameObject.SetActive(true);
        m_ScoreCanvasAnimator.SetTrigger("Enter");

        RefreshUI();

        m_AudioSource.clip = Sounds.Instance.m_GameStart;
        m_AudioSource.Play();
    }




    private void GameOver()
    {
        StopAllCoroutines();
        ReStartLevelSelector.Instance.gameObject.SetActive(false);
        m_IsSpeedFast = false;
        m_GameOver = true;
        m_EnergyBar.value = 0;
        m_LevelId = 0;
        m_Energy = 0;

        m_AsteroidSpawner.StopSpawn();
        m_AsteroidSpawner.KillAll();
        RefreshUI();
        m_SpaceShip.RefreshShields();

        m_ExplosionAnimator.SetTrigger("Explosion");

        m_AudioSource.clip = Sounds.Instance.m_FinalExplosion;
        m_AudioSource.Play();

        StartCoroutine(GameOverCoroutine());
    }

    private IEnumerator GameOverCoroutine()
    {
        yield return new WaitForSeconds(0.5f);
        ResetScraps();
        m_ScoreCanvasAnimator.gameObject.SetActive(false);
        m_SpaceShipRoot.gameObject.SetActive(false);
        m_CanvasEndScreen.gameObject.SetActive(true);
        m_ScoreEndScreenText.text= "SCORE : " + m_Score.ToString("#,0", nfi);
        m_EndScreenCenterZone.alpha = 1;
        yield return new WaitForSeconds(2.0f);
        SetUIButtonsInteractable(true);
        ReStartLevelSelector.Instance.gameObject.SetActive(true);
        m_RestartButtons[m_DifficultyLevelId].Select();
       
    }

    private void ResetScraps()
    {
        for (int i = 0; i < m_Scraps.Length; i++)
        {
            Scrap scrap = m_Scraps[i];
            scrap.Reset();
        }
    }
    private void InitializeScraps()
    {
        for (int i = 0; i < m_Scraps.Length; i++)
        {
            Scrap scrap = m_Scraps[i];
            scrap.Initialize();
        }
    }

    public AnimationCurve m_ShipRestartAnim;

    private IEnumerator RestartGameCoroutine()
    {

        for (int i = 0; i < m_Scraps.Length; i++)
        {
            Scrap scrap = m_Scraps[i];
            scrap.GetOut();
        }
        SetRestartButtonsInteractable(false);
        Vector3 startShipPos = m_SpaceShipRoot.transform.position;
        Vector3 endShipPos = Vector3.zero;
        float duration = 1;
        float p = 0;
        float timer = 0;
        while (p < 1)
        {
            timer += Time.deltaTime;
            p = timer / duration;
            float p1 = m_ShipRestartAnim.Evaluate(p);
            Vector3 shipPos = Vector3.Lerp(startShipPos, endShipPos, p1);
            m_SpaceShipRoot.transform.position = shipPos;
            m_EndScreenCenterZone.alpha = Mathf.Clamp01(1 - p*2);

            for (int i = 0; i < m_Scraps.Length; i++)
            {
                Scrap scrap = m_Scraps[i];
                scrap.m_Image.color = new Color(1, 1, 1, (1 - p));
            }

            yield return null;
        }
        SetRestartButtonsInteractable(true);
        m_SpaceShipRoot.transform.position = endShipPos;
        m_GameOver = false;
        m_LevelId = m_StartLevel;
        RefreshUI();
        m_SpaceShip.RefreshShields();
        m_CanvasEndScreen.gameObject.SetActive(false);
    }

    private void SetRestartButtonsInteractable(bool interactable)
    {
        for (int i = 0; i < 3; i++)
        {
            m_RestartButtons[i].interactable = interactable;
        }
    }

    public void RefreshUI()
    {
        m_EnergyBar.maxValue = m_MaxEnergy;

        int jaugeSpriteId = Mathf.Max(0,m_LevelId-1);
        m_JaugeImage.sprite = m_JaugeLevels[jaugeSpriteId];

        if (!m_GameStarted)
        {

        }
        else if (GameRunning)
        {
            string levelText = "Shield Level " + m_LevelId;
            if (MaxLevel && m_Energy == m_MaxEnergy)
            {
                //levelText += " (Max)";
            }
            m_LevelText.text = levelText;
        }
        else
        {
            //m_LevelText.text = "Game Over !";
            //m_RestartButton.gameObject.SetActive(true);
        }
        m_EnergyText.text = "Energy " + m_Energy + "/" + m_MaxEnergy;
        m_ScoreText.text = " Score :\n" + m_Score.ToString("#,0", nfi);
    }


    public void AddEnergy(float numEnergy)
    {
        if (numEnergy < 0)
        {
            m_LastShield.Hit(-numEnergy);
            numEnergy *= DifficultyMultiplicatorOverTime;
        }
        m_Energy += numEnergy;

        if (MaxLevel && m_Energy >= m_MaxEnergy)
        {
            m_Energy = m_MaxEnergy;
        }
        else if (m_Energy >= m_MaxEnergy)
        {
            IncreaseLevel(1);
        }
        else if (m_Energy <= 0)
        {
            if (m_LevelId == 0)
            {
                GameOver();
            }
            else
            {
                IncreaseLevel(-1);
            }
        }
        else if (m_Energy > 0 && m_LevelId==0)
        {
            m_LevelId = 1;
            m_SpaceShip.RefreshShields();
        }

        // AddScore

        if (numEnergy > 0)
        {
            m_Score += numEnergy * 1000 + numEnergy * m_ScoreBonus*100;
            m_ScoreBonus = Mathf.Clamp(m_ScoreBonus + 1,0,10);
        }
        else
        {
            m_ScoreBonus = 0;
        }


        RefreshUI();
        
    }

    public void IncreaseLevel(int levelIncrease)
    {
        int previousLevelId = m_LevelId;
        m_LevelId += levelIncrease;


        if (m_LevelId <= 0)
        {
            m_LevelId = 0;
            m_Energy = 0;
            m_MaxEnergy = CurrentLevel.m_EnergyToComplete;
            // Going to die ! Scream
            if (!m_Panic)
            {
                m_Panic = true;
                m_AudioSource.clip = Sounds.Instance.m_Panic;
                m_AudioSource.Play();
            }
        }
        else if (m_LevelId > m_Levels.Length)
        {
            m_LevelId = m_Levels.Length;
            m_Energy = CurrentLevel.m_EnergyToComplete;
        }
        else
        {
            Level previousLevel = m_Levels[Mathf.Max(0,previousLevelId-1)];
            Level newLevel = m_Levels[m_LevelId-1];

            if (levelIncrease == 1)
            {
                m_MaxLevelIdReached = m_LevelId;
                m_Energy = m_Energy - previousLevel.m_EnergyToComplete + 1;
            }
            else if (levelIncrease == -1)
            {
                m_Energy = m_Energy + newLevel.m_EnergyToComplete - 1;
            }
            m_MaxEnergy = newLevel.m_EnergyToComplete;
        }

        RefreshUI();
        m_SpaceShip.RefreshShields();
    }

    private void SetUIButtonsInteractable(bool interactable)
    {
        for (int i = 0; i < m_StartButtons.Length; i++)
        {
            m_StartButtons[i].interactable = interactable;
        }
        for (int i = 0; i < m_RestartButtons.Length; i++)
        {
            m_RestartButtons[i].interactable = interactable;
        }
    }

}
