﻿using System.Collections;
using System.Collections.Generic;
using System.Globalization;
using TMPro;
using UnityEngine;
using UnityEngine.UI;

public class AsteroidIndicator : MonoBehaviour
{
    public TMP_Text m_LevelText;
    public TMP_Text m_DistanceText;
    public TMP_Text m_MText;
    public float m_Margin = 10;
    public RectTransform m_ArrowRoot;
    public Image[] m_Arrows;
    public Image m_Hexagone;


    public Sprite[] m_LevelSprites;
    public Sprite[] m_LevelArrows;
    public Color[] m_LevelColors;

    [Header("Debug")]
    public Vector3 m_WorldPos;

    private Asteroid m_Asteroid;
    private RectTransform m_Parent;
    private RectTransform m_RectTransform;

    private Camera m_Camera;
    private CanvasGroup m_CanvasGroup;

    private float m_DistanceToAsteroid;

    NumberFormatInfo nfi = (NumberFormatInfo)CultureInfo.InvariantCulture.NumberFormat.Clone();

    private float m_Alpha = 0;
    private bool m_Destroyed = false;

    public void Initialize(Asteroid asteroid)
    {
        nfi.NumberGroupSeparator = " ";

        m_Asteroid = asteroid;
        m_Parent = transform.parent.GetComponent<RectTransform>();
        m_RectTransform = GetComponent<RectTransform>();
        m_CanvasGroup = GetComponent<CanvasGroup>();
        m_CanvasGroup.alpha = 0;
        m_Camera = Game.Instance.m_Camera;


        float offset = m_RectTransform.sizeDelta.x + m_Margin;


        float width = Screen.width; // m_Parent.sizeDelta.x;// - offset;
        float height = Screen.height; //m_Parent.sizeDelta.y;// - offset;

        float alpha = Mathf.Atan(width / height);
        float radius = width / (2 * Mathf.Sin(alpha));

        Vector3 pos = GetAngleToRectPos(m_Asteroid.m_Angle, m_RectTransform.sizeDelta.x + m_Margin);
        m_RectTransform.anchoredPosition = new Vector2(pos.x, pos.y);


        Vector3 borderScreenPos = GetAngleToRectPos(m_Asteroid.m_Angle, 0);
        

        float screenX = ((borderScreenPos.x) + width) / 2.0f;
        float screenY = ((borderScreenPos.y) + height) / 2.0f;

        Vector3 screenPos = new Vector3(screenX, screenY, 0);
        m_WorldPos = m_Camera.ScreenToWorldPoint(screenPos);

        m_WorldPos.z = 0;

        //GameObject cube = GameObject.CreatePrimitive(PrimitiveType.Cube);
        //cube.transform.position = m_WorldPos;

        m_LevelText.text = m_Asteroid.m_Level.ToString();

        // Arrow rotation
        m_ArrowRoot.localRotation = Quaternion.Euler(0,0, m_Asteroid.m_Angle);

        // Num Arrows
        float[] speedLimits = Game.Instance.m_SpeedLimitForArrows;
        //float difficultyOverTime = Game.Instance.DifficultyMultiplicatorOverTime;
        //float speed = m_Asteroid.m_Speed * difficultyOverTime;
        m_Arrows[0].gameObject.SetActive(m_Asteroid.m_Speed > speedLimits[0]);
        m_Arrows[1].gameObject.SetActive(m_Asteroid.m_Speed > speedLimits[1]);
        m_Arrows[2].gameObject.SetActive(m_Asteroid.m_Speed > speedLimits[2]);

        // Color Sprites

        int colorId = asteroid.m_Level - 1;
        m_Hexagone.sprite = m_LevelSprites[colorId];
        for (int i = 0; i < m_Arrows.Length; i++)
        {
            m_Arrows[i].sprite = m_LevelArrows[colorId];
        }
        m_LevelText.color = m_LevelColors[colorId];
        m_DistanceText.color = m_LevelColors[colorId];
        m_MText.color = m_LevelColors[colorId];



        StartCoroutine(RefreshTextCoroutine());
        StartCoroutine(FadeInCoroutine());
        //StartCoroutine(BlinkArrowCoroutine());



    }

    private IEnumerator FadeInCoroutine()
    {
        float duration = 1;
        float timer = 0;
        float p = 0;
        while (p < 1)
        {
            timer += Time.deltaTime;
            p = timer / duration;
            m_Alpha = p;
            yield return null;
        }
        m_Alpha = 1;
    }

    private IEnumerator DestroyCoroutine()
    {
        float startAlpha = m_Alpha;
        float targetAlpha = 0;
        float duration = 0.3f;
        float timer = 0;
        float p = 0;
        while (p < 1)
        {
            timer += Time.deltaTime;
            p = timer / duration;
            m_Alpha = Mathf.Lerp(startAlpha, targetAlpha, p);
            yield return null;
        }
        m_Alpha = 0;
        Destroy(gameObject);
    }

    public IEnumerator RefreshTextCoroutine()
    {
        while (true)
        {
            if(m_Destroyed) { m_DistanceToAsteroid = 0; }
            int showedDist = ((int)(m_DistanceToAsteroid * 10)) * 100;
            m_DistanceText.text = showedDist.ToString("#,0", nfi);
            float nextRefreshTime = Mathf.Min(2, m_DistanceToAsteroid / 5.0f) / m_Asteroid.m_Speed;
            yield return new WaitForSeconds(nextRefreshTime);
        }
    }
    public IEnumerator BlinkArrowCoroutine()
    {
        while (true)
        {
            float nextRefreshTime = Mathf.Clamp(m_DistanceToAsteroid / 5.0f, 0.1f, 2);
            yield return new WaitForSeconds(nextRefreshTime);
            m_ArrowRoot.gameObject.SetActive(false);
            yield return new WaitForSeconds(nextRefreshTime/8.0f);
            m_ArrowRoot.gameObject.SetActive(true);
        }
    }


    private void Update()
    {
        if (m_Asteroid == null)
        {
            Destroy(this.gameObject);
            return;
        }
        // Calculate Distance to Asteroid
        Vector3 dir = -m_Asteroid.transform.position.normalized;
        Vector3 pos = m_Asteroid.transform.position + dir * m_Asteroid.transform.localScale.x * 0.5f;
        m_DistanceToAsteroid = Vector3.Distance(m_WorldPos, pos);
        m_DistanceToAsteroid = Mathf.Max(0, (m_DistanceToAsteroid - 0.1f));

        // Calculate Alpha
        float alphaModifier = Mathf.Clamp((1 / (m_DistanceToAsteroid * 0.5f)),0.7f,1);
        m_CanvasGroup.alpha = alphaModifier * m_Alpha;
    }


    public void Destroy()
    {
        m_Destroyed = true;
        StartCoroutine(DestroyCoroutine());
    }


    private Vector3 GetAngleToRectPos(float angle, float offset)
    {
        float screenWidth = m_Parent.sizeDelta.x - offset;
        float screenHeight = m_Parent.sizeDelta.y - offset;
        float aA = Mathf.Rad2Deg * (Mathf.Atan(screenWidth / screenHeight));
        float aB = 180 - 2 * aA;
        float z1 = aA;
        float z2 = aA + aB;
        float z3 = 2 * aA + aB;
        float z4 = 3 * aA + aB;
        float z5 = 3 * aA + 2 * aB;
        float z6 = 360;

        Vector3 pos = Vector3.zero;

        if (angle >= 0 && angle < z1)
        {
            float opp = Mathf.Tan(angle * Mathf.Deg2Rad) * screenHeight / 2.0f;
            pos = new Vector3(-opp, screenHeight / 2.0f, 0);
        }
        else if (angle >= z1 && angle < z2)
        {
            float opp = Mathf.Tan((90 - angle) * Mathf.Deg2Rad) * screenWidth / 2.0f;
            pos = new Vector3(-screenWidth / 2.0f, opp);
        }
        else if (angle >= z2 && angle < z3)
        {
            float opp = Mathf.Tan((180 - angle) * Mathf.Deg2Rad) * screenHeight / 2.0f;
            pos = new Vector3(-opp, -screenHeight / 2.0f, 0);
        }
        else if (angle >= z3 && angle < z4)
        {
            float opp = Mathf.Tan((angle - 180) * Mathf.Deg2Rad) * screenHeight / 2.0f;
            pos = new Vector3(opp, -screenHeight / 2.0f, 0);
        }
        else if (angle >= z4 && angle < z5)
        {
            float opp = Mathf.Tan((angle - (270)) * Mathf.Deg2Rad) * screenWidth / 2.0f;
            pos = new Vector3(screenWidth / 2.0f, opp);
        }
        else if (angle >= z5 && angle < z6)
        {
            float opp = Mathf.Tan((360 - angle) * Mathf.Deg2Rad) * screenHeight / 2.0f;
            pos = new Vector3(opp, screenHeight / 2.0f, 0);
        }

        return pos;
    }

}
