﻿using System.Collections;
using System.Collections.Generic;
using UnityEngine;
using UnityEngine.UI;

public class Asteroid : MonoBehaviour
{
    

    [Header("Debug")]
    public Vector3 m_Direction;
    public float m_Speed;
    public int m_Level;

    public AsteroidIndicator m_AsteroidIndicator;
    public List<Shield> m_CollidedShields;

    public bool m_HasBeenAbsorbed = false;

    public float m_Distance = 0;
    public float m_Angle = 0;

    public Sprite[] m_Srites;

    private float m_StartTime;
    private float LifeTime { get { return Time.realtimeSinceStartup - m_StartTime; } }
    private Collider m_Collider;
    private Camera m_Camera;
    private Plane[] m_CameraPlanes;
    private float m_RotSpeed;
    private Image m_Image;

    private float zDepth = -0.1f;

    private void Start()
    {
        m_Collider = GetComponentInChildren<Collider>();

        m_Camera = Camera.main;
        m_CameraPlanes = GeometryUtility.CalculateFrustumPlanes(m_Camera);
        m_StartTime = Time.realtimeSinceStartup;
    }

    public void Initialize(int level=-1)
    {
        // DISTANCE
        float minDist = Game.Instance.CurrentLevel.m_MinAsteroidDist;
        float maxDist = Game.Instance.CurrentLevel.m_MaxAsteroidDist;

        m_Distance = minDist + Random.value * (maxDist - minDist);




        // LEVEL
        if (level == -1)
        {
            m_Level = GetAsteroidLevel();
        }
        else
        {
            m_Level = level;
        }

        // SPEED
        float minSpeed = Game.Instance.CurrentLevel.m_MinAsteroidSpeed;
        float maxSpeed = Game.Instance.CurrentLevel.m_MaxAsteroidSpeed;
        m_Speed = minSpeed + Random.value * (maxSpeed - minSpeed);
        float difficultyOverTime = Game.Instance.DifficultyMultiplicatorOverTime;
        m_Speed = m_Speed * difficultyOverTime * m_Distance / (5* m_Level);

        // Increase distance for fast asteroids
        m_Distance = m_Distance * Mathf.Sqrt(m_Speed);

        // POSITION
        m_Angle = Random.value * 360;
        Vector3 pos = Quaternion.AngleAxis(m_Angle, Vector3.forward) * Vector3.up * m_Distance;
        pos.z = zDepth;
        transform.position = pos;

        // DIRECTION
        Vector3 target = -transform.position;
        target.z = zDepth;

        m_Direction = target.normalized;

        // Rotation Speed

        m_RotSpeed = ((Random.value * 400)-50) / m_Level;

        // Initial Position and Size

        RefreshSize(false);

        // Spawn Indicator

        m_AsteroidIndicator = Instantiate(AsteroidSpawner.Instance.m_AsteroidIndicatorPrefab) as AsteroidIndicator;
        m_AsteroidIndicator.transform.SetParent(AsteroidSpawner.Instance.m_IndicatorsRoot, true);
        m_AsteroidIndicator.transform.localScale = Vector3.one;
        m_AsteroidIndicator.Initialize(this);

        // Sprite
        int spriteID = (int)(Random.value * m_Srites.Length);
        m_Image = GetComponentInChildren<Image>();
        m_Image.sprite = m_Srites[spriteID];

    }

    private int GetAsteroidLevel()
    {
        float[] levels = Game.Instance.CurrentLevel.m_AsteroidLevels;

        float[] scores = new float[levels.Length];
        for (int i = 0; i < scores.Length; i++)
        {
            scores[i] = Random.value * levels[i];
        }

        float highestScore = 0;
        int asteroidLevel = 0;
        for (int i = 0; i < scores.Length; i++)
        {
            float score = scores[i];
            if(score > highestScore)
            {
                asteroidLevel = i;
                highestScore = score;
            }
        }
        return asteroidLevel+1;
    }



    private void Update()
    {
        transform.Translate(m_Direction * Time.deltaTime * m_Speed, Space.World);
        transform.Rotate(Vector3.forward, m_RotSpeed * Time.deltaTime);
        m_Distance = transform.position.magnitude;

        if (LifeTime>1 && m_AsteroidIndicator != null && IsSeenByCamera())
        {
            //Debug.Log(name + " is seen by Camera");
            DestroyIndicator();
        }
    }

    private void DestroyIndicator()
    {
        m_AsteroidIndicator.Destroy();
    }

    private bool IsSeenByCamera()
    {
        return (GeometryUtility.TestPlanesAABB(m_CameraPlanes, m_Collider.bounds));
    }

    private void RefreshSize(bool moveForward=true)
    {
        float previousSize = transform.localScale.magnitude;
        transform.localScale = Vector3.one * (0.1f + m_Level * 0.6f);

        if (moveForward)
        {
            //float newSize = transform.localScale.magnitude;
            //float distForward = (previousSize - newSize) * 0.25f;

            //transform.Translate(m_Direction * distForward);
        }
    }

    public void Destroy()
    {
        AsteroidSpawner.Instance.m_Asteroids.Remove(this);
        Destroy(this.gameObject);
    }

    public void ReduceLevel(int amount)
    {
        m_HasBeenAbsorbed = true;
        m_Level = m_Level - amount;
        if (m_Level <= 0)
        {
            Destroy();
        }
        else
        {
            RefreshSize();
        }
    }

}
